<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Brand;
use App\Models\Counter;
use App\Traits\TransactionTrait;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use App\Models\Category;
use App\Models\Product;
use App\Models\Order;
use App\Models\Coupon;
use App\Models\Transection;
use App\Models\Account;
use App\Models\OrderDetail;
use App\Models\Customer;
use App\CPU\Helpers;
use Brian2694\Toastr\Facades\Toastr;
use function App\CPU\translate;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class POSController extends Controller
{
    use TransactionTrait;
    public function __construct(
        private Category $category,
        private Product $product,
        private Order $order,
        private Coupon $coupon,
        private Transection $transection,
        private Account $account,
        private OrderDetail $orderDetails,
        private Customer $customer,
        private Counter $counter,
        private Brand $brand,
    ){}

    /**
     * @param Request $request
     * @return Application|Factory|View
     */
    public function index(Request $request): Factory|View|Application
    {
        $category = $request->query('category_id', 0);
        $keyword = $request->query('search', false);
        $key = explode(' ', $keyword);

        $categories = $this->category->where('status', 1)->where('position', 0)->latest()->get();
        $counters = $this->counter->where('status', 1)->latest()->get();
        $brands = $this->brand->latest()->get();

        $products = $this->product->where('quantity', '>', 0)->active()
            ->when($request->has('category_id') && $request['category_id'] != 0, function ($query) use ($request) {
                $query->whereJsonContains('category_ids', [['id' => (string)$request['category_id']]]);
            })
            ->when($request->has('category_ids') && !empty($request->category_ids), function ($query) use ($request) {
                $query->where(function ($q) use ($request) {
                    foreach ($request->category_ids as $category_id) {
                        $q->orWhereJsonContains('category_ids', [['id' => (string)$category_id]]);
                    }
                });
            })
            ->when($request->has('subcategory_ids') && !empty($request->subcategory_ids), function ($query) use ($request) {
                $query->where(function ($q) use ($request) {
                    foreach ($request->subcategory_ids as $subcategory_id) {
                        $q->orWhereJsonContains('category_ids', [['id' => (string)$subcategory_id]]);
                    }
                });
            })
            ->when($request->has('brand_ids') && !empty($request->brand_ids), function ($query) use ($request){
                $query->whereIn('brand', $request->brand_ids);
            })
            ->when($request->has('min_price') && $request->has('max_price'), function ($query) use ($request) {
                $query->whereBetween('selling_price', [$request->min_price, $request->max_price]);
            })
            ->latest()
            ->paginate(Helpers::pagination_limit());

        $selectedCategories = $request->input('category_ids', []);
        $selectedSubcategories = $request->input('subcategory_ids', []);
        //$subcategories = $this->category->whereIn('id', $selectedSubcategories)->get();
        $subcategories = $this->category->where('position', 1)->whereIn('parent_id', $selectedCategories)->where('status', 1)->get();

        $minPrice = $this->product->where('quantity', '>', 0)->active()->min('selling_price');
        $maxPrice = $this->product->where('quantity', '>', 0)->active()->max('selling_price');

        $cartId = 'wc-' . rand(10, 1000);

        if (!session()->has('current_user')) {
            session()->put('current_user', $cartId);
        }
        if (strpos(session('current_user'), 'wc')) {
            $userId = 0;
        } else {
            $userId = explode('-', session('current_user'))[1];
        }



        if (!session()->has('cart_name')) {
            if (!in_array($cartId, session('cart_name') ?? [])) {
                session()->push('cart_name', $cartId);
            }
        }
        if (!session()->has('counter_id') && count($counters) > 0) {
            session()->put('counter_id', $counters[0]['id']);
        }
        if (count($counters) < 1){
            session()->forget('counter_id');
        }

        return view('admin-views.pos.index', compact('categories', 'products', 'cartId', 'category', 'userId', 'counters', 'minPrice', 'maxPrice', 'brands', 'selectedSubcategories', 'subcategories', 'selectedCategories'));
    }

    /**
     * @return RedirectResponse
     */
    public function clearCartIds(): RedirectResponse
    {
        session()->forget('cart_name');
        session()->forget(session('current_user'));
        session()->forget('current_user');

        return redirect()->route('admin.pos.index');
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function quickView(Request $request): JsonResponse
    {
        $product = $this->product->findOrFail($request->product_id);

        return response()->json([
            'success' => 1,
            'view' => view('admin-views.pos._quick-view-data', compact('product'))->render(),
        ]);
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function addToCart(Request $request): JsonResponse
    {
        $cartId = session('current_user');
        $userId = 0;
        $userType = 'wc';
        if (Str::contains(session('current_user'), 'sc')) {
            $userId = explode('-', session('current_user'))[1];
            $userType = 'sc';
        }
        $product = $this->product->find($request->id);
        $cart = session($cartId);
        if (session()->has($cartId) && count($cart) > 0) {
            foreach ($cart as $key => $cartItem) {
                if (is_array($cartItem) && $cartItem['id'] == $request['id']) {
                    $qty = $product->quantity - $cartItem['quantity'];
                    if ($qty == 0) {
                        return response()->json([
                            'qty' => $qty,
                            'cart_quantity' => $product->quantity,
                            'user_type' => $userType,
                            'user_id' => $userId,
                            'view' => view('admin-views.pos._cart', compact('cartId'))->render()
                        ]);
                    }
                }
            }
        }

        $data = array();
        $data['id'] = $product->id;
        $cartKeeper = [];
        $itemExist = 0;
        if (session()->has($cartId) && count($cart) > 0) {
            foreach ($cart as $key => $cartItem) {
                if (is_array($cartItem) && $cartItem['id'] == $request['id']) {
                    $cartItem['quantity'] += 1;
                    $itemExist = 1;
                }
                array_push($cartKeeper, $cartItem);
            }
        }
        session()->put($cartId, $cartKeeper);

        if ($itemExist == 0) {
            $data['quantity'] = $request['quantity'];
            $data['price'] = $product->selling_price;
            $data['name'] = $product->name;
            $data['discount'] = Helpers::discount_calculate($product, $product->selling_price);
            $data['image'] = $product->image;
            $data['tax'] = Helpers::tax_calculate($product, $product->selling_price);
            if ($request->session()->has($cartId)) {
                $keeper = [];
                foreach (session($cartId) as $item) {
                    array_push($keeper, $item);
                }
                $keeper[] = $data;
                $request->session()->put($cartId, $keeper);
            } else {
                $request->session()->put($cartId, [$data]);
            }
        }

        return response()->json([
            'user_type' => $userType,
            'user_id' => $userId,
            'view' => view('admin-views.pos._cart', compact('cartId'))->render()
        ]);
    }

    public function addToCartData(Request $request): JsonResponse
    {
        $cartId = session('current_user');
        $userId = 0;
        $userType = 'wc';
        if (Str::contains(session('current_user'), 'sc')) {
            $userId = explode('-', session('current_user'))[1];
            $userType = 'sc';
        }
        $product = $this->product->find($request->id);
        if ($product->quantity < $request->quantity){
            return response()->json([
                'qty' => 0,
                'user_type' => $userType,
                'user_id' => $userId,
                'view' => view('admin-views.pos._cart', compact('cartId'))->render()
            ]);
        }

        $cart = session($cartId);
        if (session()->has($cartId) && count($cart) > 0) {
            foreach ($cart as $key => $cartItem) {
                if (is_array($cartItem) && $cartItem['id'] == $request['id']) {
                    $qty = $product->quantity - $request->quantity;

                    if ($qty < 0) {
                        return response()->json([
                            'qty' => $qty,
                            'user_type' => $userType,
                            'user_id' => $userId,
                            'view' => view('admin-views.pos._cart', compact('cartId'))->render()
                        ]);
                    }
                }
            }
        }

        $data = array();
        $data['id'] = $product->id;
        $cartKeeper = [];
        $itemExist = 0;
        if (session()->has($cartId) && count($cart) > 0) {
            foreach ($cart as $key => $cartItem) {
                if (is_array($cartItem) && $cartItem['id'] == $request['id']) {
                    $cartItem['quantity'] = $request['quantity'];
                    $itemExist = 1;
                }
                array_push($cartKeeper, $cartItem);
            }
        }
        session()->put($cartId, $cartKeeper);

        if ($itemExist == 0) {
            $data['quantity'] = $request['quantity'];
            $data['price'] = $product->selling_price;
            $data['name'] = $product->name;
            $data['discount'] = Helpers::discount_calculate($product, $product->selling_price);
            $data['image'] = $product->image;
            $data['tax'] = Helpers::tax_calculate($product, $product->selling_price);
            if ($request->session()->has($cartId)) {
                $keeper = [];
                foreach (session($cartId) as $item) {
                    array_push($keeper, $item);
                }
                $keeper[] = $data;
                $request->session()->put($cartId, $keeper);
            } else {
                $request->session()->put($cartId, [$data]);
            }
        }

        return response()->json([
            'user_type' => $userType,
            'user_id' => $userId,
            'view' => view('admin-views.pos._cart', compact('cartId'))->render()
        ]);
    }

    /**
     * @return Application|Factory|View
     */
    public function cartItems(): Factory|View|Application
    {
        return view('admin-views.pos._cart');
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function emptyCart(Request $request): JsonResponse
    {
        $cartId = session('current_user');
        $userId = 0;
        $userType = 'wc';
        if (Str::contains(session('current_user'), 'sc')) {
            $userId = explode('-', session('current_user'))[1];
            $userType = 'sc';
        }
        session()->forget($cartId);
        return response()->json([
            'user_type' => $userType,
            'user_id' => $userId,
            'view' => view('admin-views.pos._cart', compact('cartId'))->render()
        ], 200);
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function updateQuantity(Request $request): JsonResponse
    {
        $cartId = session('current_user');
        $userId = 0;
        $userType = 'wc';
        if (Str::contains(session('current_user'), 'sc')) {
            $userId = explode('-', session('current_user'))[1];
            $userType = 'sc';
        }
        if ($request->quantity > 0) {

            $product = $this->product->find($request->key);
            $cart = session($cartId);
            $updatedQuantity = null;
            foreach ($cart as &$item) {
                if (is_array($item) && $item['id'] == $request->key) {
                        $qty = $product->quantity - $request->quantity;
                        if ($qty < 0) {
                            $item['quantity'] = $product->quantity;
                            session()->put($cartId, array_values(array_filter($cart, 'is_array')));

                            return response()->json([
                                'qty' => $qty,
                                'cart_quantity' => $product->quantity,
                                'user_type' => $userType,
                                'user_id' => $userId,
                                'view' => view('admin-views.pos._cart', compact('cartId'))->render()
                            ]);
                        }
                        $item['quantity'] = $request->quantity;
                        $updatedQuantity = $item['quantity'];
                }
            }
            unset($item);
            session()->put($cartId, array_values(array_filter($cart, 'is_array')));
            return response()->json([
                'user_type' => $userType,
                'user_id' => $userId,
                'cart_quantity' => $updatedQuantity,
                'view' => view('admin-views.pos._cart', compact('cartId'))->render()
            ], 200);
        } else {
            return response()->json([
                'upQty' => 'zeroNegative',
                'user_type' => $userType,
                'user_id' => $userId,
                'view' => view('admin-views.pos._cart', compact('cartId'))->render()
            ]);
        }
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function removeFromCart(Request $request): JsonResponse
    {
        $cartId = session('current_user');
        $userId = 0;
        $userType = 'wc';
        if (Str::contains(session('current_user'), 'sc')) {
            $userId = explode('-', session('current_user'))[1];
            $userType = 'sc';
        }
        $cart = session($cartId);
        $cartKeeper = [];
        if (session()->has($cartId) && count($cart) > 0) {
            foreach ($cart as $cartItem) {
                if (is_array($cartItem) && $cartItem['id'] != $request['key']) {
                    array_push($cartKeeper, $cartItem);
                }
            }
        }
        session()->put($cartId, $cartKeeper);

        return response()->json([
            'user_type' => $userType,
            'user_id' => $userId,
            'view' => view('admin-views.pos._cart', compact('cartId'))->render()
        ], 200);
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function updateDiscount(Request $request): JsonResponse
    {
        $cartId = session('current_user');
        $userId = 0;
        $userType = 'wc';
        if (Str::contains(session('current_user'), 'sc')) {
            $userId = explode('-', session('current_user'))[1];
            $userType = 'sc';
        }
        $cart = session($cartId, collect([]));
        if ($cart != null) {
            $totalProductPrice = 0;
            $productDiscount = 0;
            $productTax = 0;
            $extDiscount = 0;
            $couponDiscount = $cart['coupon_discount'] ?? 0;
            foreach ($cart as $ct) {
                if (is_array($ct)) {
                    $totalProductPrice += $ct['price'] * $ct['quantity'];
                    $productDiscount += $ct['discount'] * $ct['quantity'];
                    $productTax += $ct['tax'] * $ct['quantity'];
                }
            }
            $priceDiscount = 0;
            if ($request->type == 'percent') {
                $priceDiscount = (($totalProductPrice - $productDiscount - $couponDiscount) * $request->discount )/100;
            } else {
                $priceDiscount = $request->discount;
            }
            $extDiscount = $priceDiscount;
            $total = $totalProductPrice - $productDiscount + $productTax - $couponDiscount - $extDiscount;
            if ($total < 0) {
                return response()->json([
                    'extra_discount' => "amount_low",
                    'user_type' => $userType,
                    'user_id' => $userId,
                    'view' => view('admin-views.pos._cart', compact('cartId'))->render()
                ]);
            } else {
                $cart['ext_discount'] = $request->discount;
                $cart['ext_discount_type'] = $request->type;
                session()->put($cartId, $cart);

                return response()->json([
                    'extra_discount' => "success",
                    'user_type' => $userType,
                    'user_id' => $userId,
                    'view' => view('admin-views.pos._cart', compact('cartId'))->render()
                ]);
            }
        } else {
            return response()->json([
                'extra_discount' => "empty",
                'user_type' => $userType,
                'user_id' => $userId,
                'view' => view('admin-views.pos._cart', compact('cartId'))->render()
            ]);
        }
    }

    /**
     * @param Request $request
     * @return RedirectResponse
     */
    public function updateTax(Request $request): RedirectResponse
    {
        $cart = $request->session()->get('cart', collect([]));
        $cart['tax'] = $request->tax;
        $request->session()->put('cart', $cart);

        return back();
    }

    /**
     * @param $cart
     * @param $price
     * @return float|int
     */
    public function extraDisCalculate($cart, $price): float|int
    {

        if ($cart['ext_discount_type'] == 'percent') {
            $priceDiscount = ($price / 100) * $cart['ext_discount'];
        } else {
            $priceDiscount = $cart['ext_discount'];
        }

        return $priceDiscount;
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function couponDiscount(Request $request): JsonResponse
    {
        $cartId = session('current_user');
        $userId = 0;
        $userType = 'wc';
        if (Str::contains(session('current_user'), 'sc')) {
            $userId = explode('-', session('current_user'))[1];
            $userType = 'sc';
        }

        if ($userId != 0) {
            $couponLimit = $this->order->where('user_id', $userId)
                ->where('coupon_code', $request['coupon_code'])->count();

            $coupon = $this->coupon->where(['code' => $request['coupon_code']])
                ->where('user_limit', '>', $couponLimit)
                ->where('status', '=', 1)
                ->whereDate('start_date', '<=', now())
                ->whereDate('expire_date', '>=', now())->first();
        } else {
            $coupon = $this->coupon->where(['code' => $request['coupon_code']])
                ->where('status', '=', 1)
                ->where('coupon_type', '=', 'default')
                ->whereDate('start_date', '<=', now())
                ->whereDate('expire_date', '>=', now())
                ->first();
        }

        $carts = session($cartId);
        $totalProductPrice = 0;
        $productDiscount = 0;
        $productTax = 0;
        $extDiscount = 0;

        if ($coupon != null) {
            if ($carts != null) {
                foreach ($carts as $cart) {
                    if (is_array($cart)) {
                        $totalProductPrice += $cart['price'] * $cart['quantity'];
                        $productDiscount += $cart['discount'] * $cart['quantity'];
                        $productTax += $cart['tax'] * $cart['quantity'];
                    }
                }
                if ($totalProductPrice >= $coupon['min_purchase']) {
                    if ($coupon['discount_type'] == 'percent') {

                        $discount = (($totalProductPrice - $productDiscount) * ($coupon['discount']/100)) > $coupon['max_discount'] ? $coupon['max_discount'] : (($totalProductPrice - $productDiscount) * ($coupon['discount']/100)) ;
                    } else {
                        $discount = $coupon['discount'];
                    }
                    if (isset($carts['ext_discount_type'])) {
                        $extDiscount = $this->extraDisCalculate($carts, $totalProductPrice);
                    }
                    $total = $totalProductPrice - $productDiscount + $productTax - $discount - $extDiscount;
                    if ($total < 0) {
                        return response()->json([
                            'coupon' => "amount_low",
                            'user_type' => $userType,
                            'user_id' => $userId,
                            'view' => view('admin-views.pos._cart', compact('cartId'))->render()
                        ]);
                    }

                    $cart = session($cartId, collect([]));
                    $cart['coupon_code'] = $request['coupon_code'];
                    $cart['coupon_discount'] = $discount;
                    $cart['coupon_title'] = $coupon->title;
                    $request->session()->put($cartId, $cart);

                    return response()->json([
                        'coupon' => 'success',
                        'user_type' => $userType,
                        'user_id' => $userId,
                        'view' => view('admin-views.pos._cart', compact('cartId'))->render()
                    ]);
                } else {
                    return response()->json([
                        'coupon' => 'min_purchase',
                        'user_type' => $userType,
                        'user_id' => $userId,
                        'min_purchase' => $coupon['min_purchase'],
                        'view' => view('admin-views.pos._cart', compact('cartId'))->render()
                    ]);
                }
            } else {
                return response()->json([
                    'coupon' => 'cart_empty',
                    'user_type' => $userType,
                    'user_id' => $userId,
                    'view' => view('admin-views.pos._cart', compact('cartId'))->render()
                ]);
            }
        }
        return response()->json([
            'coupon' => 'coupon_invalid',
            'user_type' => $userType,
            'user_id' => $userId,
            'view' => view('admin-views.pos._cart', compact('cartId'))->render()
        ]);
    }

    /**
     * @param Request $request
     * @return RedirectResponse
     */
    public function placeOrder(Request $request): RedirectResponse
    {
        $cartId = session('current_user');
        $userId = 0;
        $userType = 'wc';
        if (Str::contains(session('current_user'), 'sc')) {
            $userId = explode('-', session('current_user'))[1];
            $userType = 'sc';
        }
        if (session($cartId)) {
            if (count(session($cartId)) < 1) {
                Toastr::error(translate('cart_empty_warning'));
                return back();
            }
        } else {
            Toastr::error(translate('cart_empty_warning'));
            return back();
        }

        $cart = session($cartId);
        $coupon_code = 0;
        $productPrice = 0;
        $orderDetails = [];
        $productDiscount = 0;
        $productTax = 0;
        $extDiscount = 0;
        $couponDiscount = $cart['coupon_discount'] ?? 0;

        $order_id = 100000 + $this->order->all()->count() + 1;
        if ($this->order->find($order_id)) {
            $order_id = $this->order->orderBy('id', 'DESC')->first()->id + 1;
        }

        $order = $this->order;
        $order->id = $order_id;
        $order->user_id = $userId;
        $order->coupon_code = $cart['coupon_code'] ?? null;
        $order->coupon_discount_title = $cart['coupon_title'] ?? null;
        $order->payment_id = $request->type;
        $order->transaction_reference = $request->transaction_reference ?? null;
        $order->comment = $request->comment ?? null;
        $order->card_number = $request->card_number ?? null;
        $order->email_or_phone = $request->email_or_phone ?? null;
        $order->counter_id = session()->has('counter_id') ? session('counter_id') : null;
        $order->created_at = now();
        $order->updated_at = now();

        foreach ($cart as $c) {
            if (is_array($c)) {
                $product = $this->product->find($c['id']);
                if ($product) {
                    $price = $c['price'];
                    $orD = [
                        'product_id' => $c['id'],
                        'product_details' => $product,
                        'quantity' => $c['quantity'],
                        'price' => $product->selling_price,
                        'tax_amount' => Helpers::tax_calculate($product, $product->selling_price),
                        'discount_on_product' => Helpers::discount_calculate($product, $product->selling_price),
                        'discount_type' => 'discount_on_product',
                        'created_at' => now(),
                        'updated_at' => now()
                    ];
                    $productPrice += $price * $c['quantity'];
                    $productDiscount += $c['discount'] * $c['quantity'];
                    $productTax += $c['tax'] * $c['quantity'];
                    $orderDetails[] = $orD;

                    $product->quantity = $product->quantity - $c['quantity'];
                    $product->order_count++;
                    $product->save();
                }
            }
        }
        $totalPrice = $productPrice - $productDiscount;

        if (isset($cart['ext_discount_type'])) {
            $extDiscount = $this->extraDisCalculate($cart, ($totalPrice - $couponDiscount));
            $order->extra_discount = $extDiscount;
        }

        $totalTaxAmount = $productTax;
        try {
            $order->total_tax = $totalTaxAmount;
            $order->order_amount = $totalPrice;
            $order->coupon_discount_amount = $couponDiscount;
            $order->collected_cash = $request->collected_cash ? $request->collected_cash : $totalPrice + $totalTaxAmount - $extDiscount - $couponDiscount;
            $order->save();

            $customer = $this->customer->where('id', $userId)->first();
            $grandTotal = $totalPrice + $totalTaxAmount - $extDiscount - $couponDiscount;

            // Handle wallet payment
            if ($userId != 0 && $request->type == 0) {
                $this->handleWalletPayment($grandTotal, $request->remaining_balance, $customer->id, $order_id);
            }

            // Handle non-wallet payment
            if ($request->type != 0) {
                $this->handleNonWalletPayment($grandTotal, $customer->id, $order_id, $request->type);
            }

            foreach ($orderDetails as $key => $item) {
                $orderDetails[$key]['order_id'] = $order->id;
            }

            $this->orderDetails->insert($orderDetails);

            session()->forget($cartId);
            session(['last_order' => $order->id]);

            Toastr::success(translate('order_placed_successfully'));
            return back();
        } catch (\Exception $e) {
            Toastr::warning(translate('failed_to_place_order'));
            return back();
        }
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function searchProduct(Request $request): JsonResponse
    {
        $request->validate([
            'name' => 'required',
        ], [
            'name.required' => translate('Product name is required'),
        ]);

        $key = explode(' ', $request['name']);
        $products = $this->product->where('quantity', '>', 0)->active()->where(function ($q) use ($key) {
            foreach ($key as $value) {
                $q->where('name', 'like', "%{$value}%");
            }
        })->orWhere(function ($q) use ($key) {
            foreach ($key as $value) {
                $q->where('product_code', 'like', "%{$value}%");
            }
        })->paginate(6);

        $countP = $products->count();

        return response()->json([
            'result' => view('admin-views.pos._search-result', compact('products'))->render(),
            'count' => $countP
        ]);
    }


    /**
     * @param Request $request
     * @return JsonResponse|void
     */
    public function searchByAddProduct(Request $request)
    {
        $request->validate([
            'name' => 'required',
        ], [
            'name.required' => translate('Product name is required'),
        ]);

        if (is_numeric($request['name'])) {
            $products = $this->product->where('quantity', '>', 0)->active()->where('product_code', $request['name'])->paginate(6);
        } else {
            $products = $this->product->where('quantity', '>', 0)->active()->where('name', $request['name'])->paginate(6);
        }

        $countP = $products->count();
        if ($countP > 0) {
            return response()->json([
                'count' => $countP,
                'id' => $products[0]->id,
            ]);
        }
    }

    /**
     * @param Request $request
     * @return Application|Factory|View
     */
    public function orderList(Request $request): Factory|View|Application
    {
        $queryParam = [];
        $search = $request['search'];
        if ($request->has('search')) {
            $orders = $this->order->latest()->where('id', 'like', "%{$search}%")->paginate(Helpers::pagination_limit())->appends($search);
        } else {
            $orders = $this->order->latest()->paginate(Helpers::pagination_limit())->appends($search);
        }

        return view('admin-views.pos.order.list', compact('orders', 'search'));
    }

    /**
     * @param $id
     * @return JsonResponse
     */
    public function generateInvoice($id)
    {
        $order = $this->order->where('id', $id)->with(['details', 'counter', 'customer', 'account', 'refund'])->first();
        return response()->json([
            'success' => 1,
            'view' => view('admin-views.pos.order.invoice', compact('order'))->render(),
        ]);
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function getCustomers(Request $request): JsonResponse
    {
        $key = explode(' ', $request['q']);
        $data = DB::table('customers')
            ->where(function ($q) use ($key) {
                foreach ($key as $value) {
                    $q->orWhere('name', 'like', "%{$value}%")
                        ->orWhere('mobile', 'like', "%{$value}%");
                }
            })->limit(6)
            ->get([DB::raw('id, IF(id <> "0",CONCAT(name,  " (", mobile ,")"), name) as text')]);

        return response()->json($data);
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function customerBalance(Request $request): JsonResponse
    {
        if ($request->customer_id == null){
            $userId = explode('-', session('current_user'))[1];
        }else{
            $userId = $request->customer_id;
        }
        $customerBalance = $this->customer->where('id', $userId)->first()->balance;
        return response()->json([
            'customer_balance' => $customerBalance
        ]);
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function removeCoupon(Request $request): JsonResponse
    {
        $cartId = ($request->user_id != 0 ? 'sc-' . $request->user_id : 'wc-' . rand(10, 1000));
        if (!in_array($cartId, session('cart_name') ?? [])) {
            session()->push('cart_name', $cartId);
        }

        $cart = session(session('current_user'));

        $cartKeeper = [];
        if (session()->has(session('current_user')) && count($cart) > 0) {
            foreach ($cart as $cartItem) {

                array_push($cartKeeper, $cartItem);
            }
        }
        $tempCartName = [];
        if (session('current_user') != $cartId) {
            foreach (session('cart_name') as $cart_name) {
                if ($cart_name != session('current_user')) {
                    $tempCartName[] = $cart_name;
                }
            }
            session()->put('cart_name', $tempCartName);
        }
        session()->put('cart_name', $tempCartName);
        session()->forget(session('current_user'));
        session()->put($cartId, $cartKeeper);
        session()->put('current_user', $cartId);
        $userId = explode('-', session('current_user'))[1];
        $currentCustomer = '';
        if (explode('-', session('current_user'))[0] == 'wc') {
            $currentCustomer = 'Walking Customer';
        } else {
            $current = $this->customer->where('id', $userId)->first();
           // $currentCustomer = $current->name . ' (' . $current->mobile . ')';
            $currentCustomer = $current->name;
        }

        return response()->json([
            'cart_nam' => session('cart_name'),
            'current_user' => session('current_user'),
            'current_customer' => $currentCustomer,
            'current_customer_phone' => $current->mobile ?? '',
            'current_customer_balance' => $current->balance ?? 0,
            'view' => view('admin-views.pos._cart', compact('cartId'))->render()
        ]);
    }

    public function selectedCustomer()
    {
        $userId = explode('-', session('current_user'))[1];

        if (explode('-', session('current_user'))[0] == 'wc') {
            $currentCustomer = 'Walking Customer';
            $user_type = 0;
        } else {
            $user_type = 1;
            $current = $this->customer->where('id', $userId)->first();
            $currentCustomer = $current->name;
        }
        return response()->json([
            'cart_nam' => session('cart_name'),
            'current_user' => session('current_user'),
            'current_customer' => $currentCustomer,
            'current_customer_phone' => $current->mobile ?? '',
            'current_customer_balance' => $current->balance ?? 0,
            'user_type' => $user_type,
            'customer_id' => $userId,
        ]);
    }

    /**
     * @param Request $request
     * @return RedirectResponse
     */
    public function changeCart(Request $request): RedirectResponse
    {
        session()->put('current_user', $request->cart_id);
        return redirect()->route('admin.pos.index');
    }

    /**
     * @param Request $request
     * @return RedirectResponse
     */
    public function newCartId(Request $request): RedirectResponse
    {
        $cartId = 'wc-' . rand(10, 1000);
        session()->put('current_user', $cartId);
        if (!in_array($cartId, session('cart_name') ?? [])) {
            session()->push('cart_name', $cartId);
        }

        return redirect()->route('admin.pos.index');
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function getCartIds(Request $request): JsonResponse
    {
        $cartId = session('current_user');
        $userId = 0;
        $userType = 'wc';
        if (Str::contains(session('current_user'), 'sc')) {
            $userId = explode('-', session('current_user'))[1];
            $userType = 'sc';
        }
        $cart = session($cartId);
        $cartKeeper = [];
        if (session()->has($cartId) && count($cart) > 0) {
            foreach ($cart as $cartItem) {
                $cartKeeper[] = $cartItem;
            }
        }
        session()->put(session('current_user'), $cartKeeper);
        $userId = explode('-', session('current_user'))[1];
        $currentCustomer = '';
        if (explode('-', session('current_user'))[0] == 'wc') {
            $currentCustomer = 'Walking Customer';
        } else {
            $current = $this->customer->where('id', $userId)->first();
           // $currentCustomer = $current->name . ' (' . $current->mobile . ')';
            $currentCustomer = $current->name;

        }
        return response()->json([
            'current_user' => session('current_user'),
            'cart_nam' => session('cart_name') ?? '',
            'current_customer' => $currentCustomer,
            'current_customer_phone' => $current->mobile ?? '',
            'current_customer_balance' => $current->balance ?? 0,
            'user_type' => $userType,
            'user_id' => $userId,
            'view' => view('admin-views.pos._cart', compact('cartId'))->render()
        ]);
    }

    public function changeCounter(Request $request)
    {
        session()->put('counter_id', $request->counter_id);
        return response()->json();
    }

    public function getSubcategories(Request $request)
    {
        $categoryIds = $request->category_ids;
        $subcategories = [];
        if ($categoryIds != null) {
            $subcategories = $this->category->where('position', 1)
                ->whereIn('parent_id', $categoryIds)
                ->where('status', 1)
                ->get();
        }

        return response()->json([
            'subcategories' => $subcategories
        ]);
    }

    public function cancelHoldOrder(Request $request)
    {
        $cartId = $request['cart_id'];
        $sessionCarts = session()->has('cart_name') ? session('cart_name') : [];

        foreach ($sessionCarts as $key => $cart) {
            if ($cart == $cartId) {
                unset($sessionCarts[$key]);
            }
        }

        if (count($sessionCarts) < 1) {
            session()->forget('cart_name');
        } else {
            session()->put('cart_name', $sessionCarts);
        }

        session()->forget($cartId);

        if (session('current_user') == $cartId) {
            session()->forget('current_user');
        }

        $newCartId = 'wc-' . rand(10, 1000);

        if (!session()->has('current_user')) {
            session()->put('current_user', $newCartId);
        }

        if (!session()->has('cart_name')) {
            session()->put('cart_name', [$newCartId]);
        } elseif (!in_array($newCartId, session('cart_name'))) {
            session()->push('cart_name', $newCartId);
        }

        return redirect()->route('admin.pos.index');
    }

    public function getCoupon() {
        $userId = 0;
        if (Str::contains(session('current_user'), 'sc')) {
            $userId = explode('-', session('current_user'))[1];
        }

        if ($userId != 0) {
            $coupons = $this->coupon
                ->where('status', 1)
                ->whereDate('start_date', '<=', now())
                ->whereDate('expire_date', '>=', now())
                ->get()
                ->filter(function ($coupon) {
                    $orderCount = $this->order
                        ->where('coupon_code', $coupon->code)
                        ->count();
                    return $orderCount < $coupon->user_limit;
                });
        } else {
            $coupons = $this->coupon
                ->where('status', '=', 1)
                ->where('coupon_type', '=', 'default')
                ->whereDate('start_date', '<=', now())
                ->whereDate('expire_date', '>=', now())
                ->get();
        }
        return response()->json([
            'total_coupon' => $coupons->count(),
            'view' => view('admin-views.pos._get-coupon', compact('coupons'))->render()
        ]);

    }
}
